# Part of the ROBOID project - http://hamster.school
# Copyright (C) 2016 Kwang-Hyun Park (akaii@kw.ac.kr)
# 
# This library is free software; you can redistribute it and/or
# modify it under the terms of the GNU Lesser General Public
# License as published by the Free Software Foundation; either
# version 2.1 of the License, or (at your option) any later version.
# 
# This library is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# Lesser General Public License for more details.
# 
# You should have received a copy of the GNU Lesser General
# Public License along with this library; if not, write to the
# Free Software Foundation, Inc., 59 Temple Place, Suite 330,
# Boston, MA  02111-1307  USA

import time
from timeit import default_timer as timer


class Runner(object):
    _added = []
    _removed = []
    _robots = []
    _thread = None
    _required = 0
    _checked = 0
    _start_flag = False
    _execute = None
    _evaluate = None
    _evaluate_result = False

    @staticmethod
    def shutdown():
        robots = Runner._robots
        Runner._robots = []
        for robot in robots:
            robot.dispose()

        Runner._running = False
        thread = Runner._thread
        Runner._thread = None
        if thread:
            thread.join()

    @staticmethod
    def register_robot(robot):
        Runner._added.append(robot)

    @staticmethod
    def unregister_robot(robot):
        Runner._removed.append(robot)

    @staticmethod
    def register_required():
        Runner._required += 1

    @staticmethod
    def register_checked():
        Runner._checked += 1

    @staticmethod
    def set_executable(execute):
        Runner._execute = execute

    @staticmethod
    def wait(milliseconds):
        if isinstance(milliseconds, (int, float)) and milliseconds > 0:
            timeout = timer() + milliseconds / 1000.0
            while timer() < timeout:
                time.sleep(0.000000001)

    @staticmethod
    def wait_until_ready():
        while Runner._checked < Runner._required:
            time.sleep(0.01)

    @staticmethod
    def wait_until(evaluate):
        Runner._evaluate_result = False
        Runner._evaluate = evaluate
        while Runner._evaluate_result == False:
            time.sleep(0.01)

    @staticmethod
    def _run():
        try:
            target_time = timer()
            while Runner._running:
                if timer() > target_time:
                    added = Runner._added
                    removed = Runner._removed
                    robots = Runner._robots

                    if len(added) > 0:
                        for robot in added:
                            robots.append(robot)
                        Runner._added = []
                    if len(removed) > 0:
                        for robot in removed:
                            robots.remove(robot)
                        Runner._removed = []

                    for robot in robots:
                        robot._update_sensory_device_state()

                    if Runner._evaluate:
                        result = False
                        try:
                            result = Runner._evaluate.__func__()
                        except:
                            try:
                                result = Runner._evaluate()
                            except:
                                Runner._evaluate = None
                        Runner._evaluate_result = result
                        if result:
                            Runner._evaluate = None

                    if Runner._execute:
                        try:
                            Runner._execute.__func__()
                        except:
                            try:
                                Runner._execute()
                            except:
                                Runner._execute = None

                    for robot in robots:
                        robot._request_motoring_data()
                    for robot in robots:
                        robot._update_motoring_device_state()
                    for robot in robots:
                        robot._notify_motoring_device_data_changed()

                    target_time += 0.02
                time.sleep(0.000000001)
        except:
            pass

    @staticmethod
    def start():
        import threading
        if Runner._start_flag == False:
            Runner._start_flag = True
            Runner._running = True
            thread = threading.Thread(target=Runner._run)
            Runner._thread = thread
            thread.daemon = True
            thread.start()